/*
 * Summary: drivercom.c
 * manages the initial driver communication port
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     Public Domain
 */

#include <mos/drivercom.h>
#include <string.h>

/* Packet format:
   dword destination (write last)
   dword source
   dword size of payload (size of packet - 12)
   size bytes
 */

static int drivername;

/* Function: drv_sendmessage
 * sends a message over communication port
 *
 * in:
 *     dest   - address of the receiving object
 *     size   - size of the data to be send
 *     buffer - pointer to the data to send
 *
 */
void drv_sendmessage(unsigned int dest, unsigned int size, char * buffer)
{
    int outptr;
    char * bufferbase = &(((char *) _MOS_bootinfo)[0x800]);
    unsigned int realsize = size + 12 + (4-(size%4))%4;
    __asm__ __volatile__("lock xaddl %%eax, (%%ecx)" : "=a"(outptr) : "a"(realsize), "c"(&(_MOS_bootinfo->writept)) : "memory");
    outptr = outptr & 0x7ff;
    int outptr2 = (outptr + 4) & 0x7ff;
    *((int *)(bufferbase+outptr2)) = drivername;
    outptr2 = (outptr2 + 4) & 0x7ff;
    *((int *)(bufferbase+outptr2)) = size;
    outptr2 = (outptr2 + 4) & 0x7ff;

    if (outptr2 + size > 0x800)
    {
       unsigned int sizea = 0x800 - outptr2;
       memcpy(bufferbase+outptr2, buffer, sizea);
       memcpy(bufferbase, &(buffer[sizea]), size - sizea);
    }
    else memcpy(bufferbase+outptr2, buffer, size);

    *((int *)(bufferbase+0+outptr)) = dest;
}


/* Function: drv_setname
 *
 */
void drv_setname(int service, int index)
{
     drivername = service << 16 | (index & 0xffff);
}

/* Function: drv_peekmessage
 * Checks if a message is waiting
 *
 * in:
 *     nothing
 *
 * out:
 *     return - size of the waiting message, 0 if none present
 */
int drv_peekmessage(void)
{
    int nextmsg = ((int)_MOS_bootinfo->readpt) & 0x7ff;
    char * bufferbase = &(((char *) _MOS_bootinfo)[0x800]);
    int *dst = (int*) &(bufferbase[nextmsg]);
    if (*dst != drivername) return 0;
    nextmsg = (nextmsg + 8) & 0x7ff;
    int *size = (int*) &(bufferbase[nextmsg]);
    return *size;
}

/* Function: drv_readmessage
 * Reads the message from the queue
 *
 * in:
 *     dest - pointer to buffer to hold a copy
 *
 * out:
 *     return - address of the sending object
 */
int drv_readmessage(char * dest)
{
    unsigned int nextmsg = ((unsigned int)_MOS_bootinfo->readpt) & 0x7ff;
    char * bufferbase = &(((char *) _MOS_bootinfo)[0x800]);
    int *dst = (int*) &(bufferbase[nextmsg]);
    if (*dst != drivername) return 0;

    *dst = 0;
    nextmsg = (nextmsg + 4) & 0x7ff;
    unsigned int src = *((unsigned int*) &(bufferbase[nextmsg]));
    *((int*) &(bufferbase[nextmsg])) = 0;
    nextmsg = (nextmsg + 4) & 0x7ff;
    unsigned int size = *((unsigned int*) &(bufferbase[nextmsg]));
    *((unsigned int*) &(bufferbase[nextmsg])) = 0;
    nextmsg = (nextmsg + 4) & 0x7ff;

    if (nextmsg + size > 0x800)
    {
        unsigned int sizea = 0x800 - nextmsg;
        memcpy(dest, bufferbase + nextmsg, sizea);
        memcpy(&dest[sizea], bufferbase, size-sizea);
        memset(bufferbase+nextmsg, 0, sizea);
        memset(bufferbase, 0, size-sizea);
    }
    else
    {
        memcpy(dest, bufferbase + nextmsg, size);
        memset(bufferbase+nextmsg, 0, size);
    }

    unsigned int realsize = size + 12 + (4-(size%4))%4;
    __asm__ __volatile__("lock addl %%eax, (%%ecx)" : : "a"(realsize), "c"(&(_MOS_bootinfo->readpt)) : "memory");

    return src;

}
