/*
 * Summary: hosted_comm.c
 * Communication primitives for external platforms, used to emulate the native event train.
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     Public Domain
 */

#include <mos/drivercom.h>
#include <string.h>
#include <stdint.h>
#include <stdlib.h>

/* Packet header
 *
 */

typedef struct message_t
{
    struct message_t * next;
    char * data;
    size_t size;
    uint32_t source;
    uint32_t destination;
} message_t;

/* Packet format:
   dword destination (write last)
   dword source
   dword size of payload (size of packet - 12)
   size bytes
 */

static uint32_t drivername;
static message_t * queue_head = NULL;
static message_t * queue_tail = NULL;


/* Function: drv_sendmessage
 * sends a message over communication port
 *
 * in:
 *     dest   - address of the receiving object
 *     size   - size of the data to be send
 *     buffer - pointer to the data to send
 *
 */
void drv_sendmessage(unsigned int dest, unsigned int size, char * buffer)
{
    message_t * msg  = (message_t *) malloc(sizeof(message_t));
    char * copy      = (char *) malloc(size);
    memcpy(copy, buffer, size);
    msg->data        = copy;
    msg->size        = size;
    msg->source      = drivername;
    msg->destination = dest;
    msg->next        = NULL;

    if (queue_tail)
    {
        queue_tail->next = msg;
	queue_tail = msg;
    }
    else
    {
        queue_head = queue_tail = msg;
    }
}


/* Function: drv_setname
 *
 */
void drv_setname(int service, int index)
{
     drivername = service << 16 | (index & 0xffff);
}

/* Function: drv_peekmessage
 * Checks if a message is waiting
 *
 * in:
 *     nothing
 *
 * out:
 *     return - size of the waiting message, 0 if none present
 */
int drv_peekmessage(void)
{
    if (!queue_head) return 0;
    return queue_head->size;
}

/* Function: drv_readmessage
 * Reads the message from the queue
 *
 * in:
 *     dest - pointer to buffer to hold a copy
 *
 * out:
 *     return - address of the sending object
 */
int drv_readmessage(char * dest)
{
    memcpy(dest, queue_head->data, queue_head->size);
    int sender = queue_head->source;

    message_t * next = queue_head->next;
    free(queue_head->data);
    free(queue_head);

    queue_head = next;
    if (next == NULL) queue_tail = NULL;

    return sender;
}

/* Function: drv_peekmessageany
 * Checks if any message is waiting, including messages to others
 *
 * in:
 *     target - pointer to a variable to hold the destination address
 *
 * out:
 *     return - size of the waiting message, 0 if none present
 *     target - if not null, set to the destination address
 */
int drv_peekmessageany(int * target)
{
    if (!queue_head) return 0;
    if (target) *target = queue_head->destination;
    return queue_head->size;
}

/* Function: drv_peeksender
 * Checks the sender of the message currently at the front of the queue
 *
 * in:
 *     none
 *
 * out:
 *     return - sender of the waiting message, 0 if none present
 */
int drv_peeksender()
{
    if (!queue_head) return 0;
    return queue_head->source;
}

/* Function: drv_readmessageany
 * Reads a message from the queue, independent of target
 *
 * in:
 *     dest - pointer to buffer to hold a copy
 *
 * out:
 *     return - address of the sending object
 */
int drv_readmessageany(char * dest)
{
    return drv_readmessage(dest);
}


/* Function: drv_sendmessageany
 * sends a message over communication port
 *
 * in:
 *     dest   - address of the receiving object
 *     src    - sender address to attach to the message
 *     size   - size of the data to send
 *     buffer - pointer to the data to send
 *
 */
void drv_sendmessageany(unsigned int dest, unsigned int src, unsigned int size, char * buffer)
{
    int tempname = drivername;
    drivername = src;
    drv_sendmessage(dest, size, buffer);
    drivername = tempname;
}

