/**
 * Summary: idlemit_udidef.c
 * IDL-to-code emitter for prototyping the standards header
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#include "idltool.h"
#include <stdio.h>
#include <ctype.h>

static int idl_emit_udidef_cb(FILE * out, IDL_PARAM * tree)
{
    if (!tree || tree->param_type != DATA_TUPLE >> 8)
    {
        fprintf(stderr, "Error: UDI requires first parameter to be a cb struct");
        return 1;
    }
    
    fprintf(out, "#ifndef _idlgen_cb_def_%s\n"
                 "#define _idlgen_cb_def_%s\n",
            tree->param_customtype,
            tree->param_customtype);
    fprintf(out, "typedef struct %s\n"
                 "{\n"
                 "    udi_cb_t gcb;\n",
            tree->param_customtype);
    
    IDL_PARAM * child = tree->child;
    while(child)
    {
        switch (child->param_type)
        {
            case (DATA_8 >> 8):  fprintf(out, "    udi_ubit8_t %s;\n",  child->param_name); break;
            case (DATA_16 >> 8): fprintf(out, "    udi_ubit16_t %s;\n", child->param_name); break;
            case (DATA_32 >> 8): fprintf(out, "    udi_ubit32_t %s;\n", child->param_name); break;
            case (DATA_STR >> 8):fprintf(out, "    udi_buf_t * %s;\n",  child->param_name); break;
            
            default:
                fprintf(stderr, "BUG: unknown data type %i", child->param_type);
                return 2;
        }
        child = child->next;
    }
    
    fprintf(out, "} %s;\n"
                 "#endif\n",
            tree->param_customtype);
    
    return 0;
}

static int idl_emit_udidef_arguments(FILE * out, IDL_PARAM * tree)
{
    if (!tree || tree->param_type != DATA_TUPLE >> 8)
    {
        fprintf(stderr, "Error: UDI requires first parameter to be a cb struct");
        return 1;
    }

    fprintf(out, "%s *cb", tree->param_customtype); 

    tree = tree->next;
    while (tree)
    {
        switch (tree->param_type)
        {
            case (DATA_8 >> 8):  fprintf(out, ", udi_ubit8_t %s",  tree->param_name); break;
            case (DATA_16 >> 8): fprintf(out, ", udi_ubit16_t %s", tree->param_name); break;
            case (DATA_32 >> 8): fprintf(out, ", udi_ubit32_t %s", tree->param_name); break;
            case (DATA_STR >> 8):fprintf(out, ", udi_buf_t * %s",  tree->param_name); break;

            default:
                fprintf(stderr, "BUG: unknown data type %i", tree->param_type);
                return 2;
        }
        tree = tree->next;
    }
    
    return 0;
}

static int idl_emit_udidef_proto(FILE * out, IDL_CHANNEL * data)
{
    IDL_MESSAGE * message = data->messages;
    while (message)
    {
        // create CBs
        int retval = idl_emit_udidef_cb(out, message->params);       
        if (retval) return retval;
        
        fprintf(out, "typedef void udi_%s_op_t(",
                message->message_name);

        retval = idl_emit_udidef_arguments(out, message->params);
        if (retval) return retval;
        
        fprintf(out, ");\n");
        fprintf(out, "udi_%s_op_t udi_%s;\n\n", message->message_name, message->message_name);

        message = message->next;
    }
    
    message = data->messages;
    fprintf(out, "typedef struct udi_%s_ops_t\n"
                 "{\n",
            data->interface_name);
    
    
    while (message)
    {
        fprintf(out, "    udi_%s_op_t * %s_op;\n",
                message->message_name,
                message->message_name);
        message = message->next;
    }
    
    fprintf(out, "} udi_%s_ops_t;\n\n", data->interface_name);
    
    return 0;
}


int idl_emit_udidef(FILE * out, IDL_CHANNEL * data)
{
    fprintf(out, "// Automatically generated by idltool\n// Do not modify this file, but copy-paste or include as appropriate\n\n");

    if (!data)
    { 
        fprintf(stderr, "Error: no protocols defined\n");
        return 1;
    }

    while (data)
    {
        int retval = idl_emit_udidef_proto(out, data);
        if (retval) return retval;
        data = data->next;
    }

    fprintf(out, "\n");

    return 0;
}
