/**
 * Summary: idlemit_udirx.c
 * IDL-to-code emitter for message handling and UDI event injection
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#include "idltool.h"
#include <stdio.h>
#include <ctype.h>

static void write_ucase(FILE * out, const char * string)
{
    while (*string)
    {
        fputc(toupper(*string++), out);
    }    
}

static void write_lcase(FILE * out, const char * string)
{
    while (*string)
    {
        fputc(tolower(*string++), out);
    }    
}

int idl_emit_udirx_sizecount(FILE * out, IDL_PARAM * param)
{
    while (param)
    {
        switch (param->param_type << 8)
        {
            case DATA_STR:            
                fprintf(out, " + _convdata->%s", param->param_name);
                break;
            
            case DATA_TUPLE:            
                idl_emit_udirx_sizecount(out, param->child);
                break;

            default:
                break;
        }
        param = param->next;
    }
    return 0;
}

int idl_emit_udirx_dispatch(FILE * out, IDL_CHANNEL * data, IDL_MESSAGE * message)
{
    /*
    fprintf(out, "typedef struct udienv_wrapf_%s\n"
                 "{\n"
                 "    udi_%s_op_t * entrypoint;\n"
                 "} udienv_wrapf_%s_t;\n\n",
            message->message_name,
            message->message_name,
            message->message_name);
	    */
    
    fprintf(out, "int udienv_dispatch_%s(void * user, unsigned int sender, unsigned int length, udi_ubit8_t * data)\n"
                 "{\n",
                message->message_name);
    fprintf(out, "bochs_puts(\"[dispatch %s]\");", 
                message->message_name);
    
    fprintf(out, "    if (length < sizeof(udienv_marshal_%s_t))\n"
                 "    {\n"
                 "        return 1;\n"
                 "    }\n\n",
            message->message_name);
    
    fprintf(out, "    udienv_marshal_%s_t * _convdata = (udienv_marshal_%s_t *) data;\n",
            message->message_name,
            message->message_name
           );
    fprintf(out, "    udienv_register_%s_t * _user = (udienv_register_%s_t*)user;\n"
                 "    udi_%s_op_t * _entrypoint = _user->entrypoint_%s;\n", 
            data->interface_name,
	    data->interface_name,
	    message->message_name,
	    message->message_name
           );
    
    // real buffer size
    fprintf(out, "    udi_size_t _baseoffset = sizeof(udienv_marshal_%s_t);\n"
                 "    udi_size_t _totalbytes = _baseoffset", 
            message->message_name
           );    
    idl_emit_udirx_sizecount(out, message->params);
    fprintf(out, ";\n"
                 "    if (_totalbytes != length)\n"
                 "    {\n"
                 "        return 1;\n"
                 "    }\n\n");
    
    IDL_PARAM * param = message->params;
    IDL_PARAM * cb = param;
    if (param == NULL || param->param_type << 8 != DATA_TUPLE)
    {
        fprintf(stderr, "Expected a CB as the first argument on an UDI interface\n");
        return 1;
    }    
    
    fprintf(out, "    %s * cb = (%s *) _udi_cb_enter(_user->region, sizeof(%s));\n",
            param->param_customtype,
            param->param_customtype,
            param->param_customtype
                );
    
    fprintf(out, "    mos_system_cb_t * syscb = _udi_cb_system(UDI_GCB(cb));\n"
                 "    syscb->origin = sender;\n"
                 "    syscb->requestindex = _convdata->_reply_id;\n\n");                 
    
    IDL_PARAM * embed = cb->child;
    while (embed)
    {
        switch (embed->param_type << 8)
        {
            case DATA_8:
            case DATA_16:
            case DATA_32:
                fprintf(out, "    cb->%s = _convdata->%s;\n", embed->param_name, embed->param_name);
                break;
                
            case DATA_STR:
                fprintf(out, "    cb->%s = _udi_buf_enter(data + _baseoffset, _convdata->%s);\n"
                             "    _baseoffset += _convdata->%s;\n", 
                        embed->param_name, embed->param_name, embed->param_name);
                break;
            
            default:
                fprintf(stderr, "BUG: unsupported type %i in argument %s\n", embed->param_type, embed->param_name);
                return 1;                
        }
        embed = embed->next;
    }
    
    IDL_PARAM * args = param->next;
    while(args)
    {
        switch (args->param_type << 8)
        {
            case DATA_STR:
                fprintf(out, "    udi_buf_t * _arg_%s = _udi_buf_enter(data + _baseoffset, _convdata->%s);\n"
                             "    _baseoffset += _convdata->%s;\n", 
                        args->param_name, args->param_name, args->param_name);
                break;
            
            case DATA_8:
            case DATA_16:
            case DATA_32:
                // not variable length
                break;
                
            default:
                fprintf(stderr, "BUG: unsupported type %i in argument %s\n", args->param_type, args->param_name);
                return 1;
        }
        args = args->next;
    }
    
    args = param->next;
    fprintf(out, "    _entrypoint(cb");
    while(args)
    {
        switch (args->param_type << 8)
        {
            case DATA_8:
            case DATA_16:
            case DATA_32:
                fprintf(out, ", _convdata->%s", args->param_name);
                break;

            case DATA_STR:
                fprintf(out, ", _arg_%s", args->param_name);
                break;

            default:
                fprintf(stderr, "BUG: unsupported type %i in argument %s\n", args->param_type, args->param_name);
                return 1;
        }
        args = args->next;
    }
    fprintf(out, ");\n\n");
    
    fprintf(out, "    return 0;\n");
    fprintf(out, "}\n\n");        
    
    return 0;    
}

int idl_emit_udirx_struct(FILE * out, IDL_CHANNEL * data)
{    
    // grand struct
    IDL_MESSAGE * base = data->messages;
    fprintf(out, "typedef struct udienv_register_%s_t\n"
                 "{\n"
		 "    mos_region_t * region;\n",
        data->interface_name);    
    
    while (base)
    {
        fprintf(out, "    udi_%s_op_t * entrypoint_%s;\n", 
                base->message_name,
                base->message_name);
        base = base->next;
    }
    
    fprintf(out, "} udienv_register_%s_t; \n\n",
            data->interface_name);

    return 0;
}

int idl_emit_udirx_register(FILE * out, IDL_CHANNEL * data)
{
    fprintf(out, "void udienv_register_%s(MOS_EVENTHANDLER_ROOT * eventhandles, udi_%s_ops_t * ops, mos_region_t * region)\n"
                 "{\n"
                 "    udienv_register_%s_t * metadata = (udienv_register_%s_t *) malloc(sizeof(udienv_register_%s_t));\n"
                 "    udi_assert(metadata != NULL);\n\n"
		 "    metadata->region = region;\n",
            data->interface_name,
            data->interface_name,
            data->interface_name,
            data->interface_name,
            data->interface_name);
   
    
    IDL_MESSAGE * base = data->messages;
    while (base)
    {
        fprintf(out, "    metadata->entrypoint_%s = ops->%s_op;\n",
            base->message_name,
            base->message_name);
        
        fprintf(out, "    mos_resize_insert(eventhandles, ");
        write_ucase(out, base->message_name);
        fprintf(out, "_MESSAGE_ID, &udienv_dispatch_%s, metadata);\n",
            base->message_name);
        
        
        base = base->next;        
    }
    
    
    fprintf(out, "}\n\n");
    
    return 0;
}

    

int idl_emit_udirx(FILE * out, IDL_CHANNEL * origdata)
{
    fprintf(out, "// Automatically generated by idltool\n// Do not modify\n\n");    

    if (!origdata)
    { 
        fprintf(stderr, "Error: no protocols defined\n");
        return 1;
    }
    
    fprintf(out, "#include \"");    
    write_lcase(out, origdata->interface_name);
    fprintf(out, ".h\"\n");
    
    fprintf(out, "#include \"mos/");
    write_lcase(out, origdata->interface_name);    
    fprintf(out, "_ids.h\"\n\n");    

    fprintf(out, "#include <udi_env.h>\n");
    fprintf(out, "#include <stdlib.h>\n");
    fprintf(out, "#include <mos/event.h>\n");
    fprintf(out, "void bochs_puts(const char * s);\n");
    
    IDL_CHANNEL * data = origdata;
    while (data)
    {
        int retval;
       
        retval = idl_emit_udirx_struct(out, data);
	if (retval) return retval;

        IDL_MESSAGE * message = data->messages;
        while (message)
        {
            retval = idl_emit_udirx_dispatch(out, data, message);
            if (retval) return retval;            
            
            message = (IDL_MESSAGE *) message->next;
        }        
        retval = idl_emit_udirx_register(out, data);
        if (retval) return retval;
            
        data = (IDL_CHANNEL *) data->next;
    }    
    
    return 0;
}
