#include "idltool.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

static char * sinkfile(FILE * file)
{
    char * buffer = malloc(512);
    int totalsize = 512;
    int offset = 0;
    int read_bytes = 0;
    
    do
    {
        if (buffer == NULL)
        {
            fprintf(stderr, "out of memory\n");
            return NULL;
        }
        
        int chunk = totalsize - offset;
        if (chunk > 4096) chunk = 4096;
        
        read_bytes = fread(buffer + offset, 1, totalsize - offset, file);
        if (read_bytes >= 0)
        {
            offset += read_bytes;
            if (offset == totalsize)
            {
                totalsize = totalsize * 2;
                char * newbuffer = realloc(buffer, totalsize);
                if (newbuffer == NULL)
                {
                    free(buffer);
                    buffer = NULL;
                }
		else
		{
                    buffer = newbuffer;
		}
            }
        }
        else
        {
            fprintf(stderr, "read error: %s\n", strerror(errno));
            free(buffer);
            buffer = NULL;
        }
    }
    while (read_bytes > 0);
 
    return buffer;
}


int main(int argc, const char ** argv)
{
    const char * fn_in;
    const char * fn_out;

    int result = 0;
    
    // basic usage check
    if (argc != 3 && argc != 4)
    {
        fprintf(stderr, "Error: invalid arguments\nUsage: %s emitter input [output]\n", argv[0]);
        return(1);
    }

    int emit = 0;
    if (strcmp(argv[1], "bi") == 0)
        emit = EMIT_B_HEADER;
    else if (strcmp(argv[1], "bas") == 0)
        emit = EMIT_B_IMPL;    
    else if (strcmp(argv[1], "pp") == 0)
        emit = EMIT_PRETTY;    
    else if (strcmp(argv[1], "udih") == 0)
        emit = EMIT_UDI_HEADER;
    else if (strcmp(argv[1], "udidef") == 0)
        emit = EMIT_UDI_STANDARD;
    else if (strcmp(argv[1], "udirx") == 0)
        emit = EMIT_UDI_RECEIVER;
    else if (strcmp(argv[1], "uditx") == 0)
        emit = EMIT_UDI_TRANSMITTER;
    else
    {
        fprintf(stderr, "Error: unknown emitter %s\nKnown emitters are: bi bas udih (todo: c h hpp uditx udirx udiq)\n", argv[2]);
        return(1);
    }

    // open input
    fn_in = argv[2];
    FILE * fin = fopen(fn_in, "r");
    if (!fin)
    {
        fprintf(stderr, "Error opening %s: %s\n", fn_in, strerror(errno));
        return(1);
    }

    // parse
    char * data = sinkfile(fin);
    fclose(fin);
    if (!data) return 1;
    
    IDL_LEX * lex = idl_lexer(data);
    free(data);
    if (!lex)
        return 1;
    
    IDL_CHANNEL * parse = idl_parser(lex);
    idl_freelex(lex);
    
    if (!parse)
        return 1;        

    // open output if specified, stdout otherwise
    FILE * fout = NULL;
    if (argc == 4)
    {
        fn_out = argv[3];
        fout = fopen(fn_out, "w");
    }
    else
    {
        fn_out = "stdout";
        fout = stdout;
    }
    if (!fout)
    {
        fprintf(stderr, "Error opening %s: %s\n", fn_in, strerror(errno));
        idl_freeparse(parse);
	return(1);
    }
        
    if (emit == EMIT_B_HEADER)
    {
        result = idl_emit_bi(fout, parse);
    }
    else if (emit == EMIT_B_IMPL)
    {
        result = idl_emit_bas(fout, parse);
    }
    else if (emit == EMIT_UDI_HEADER)
    {
        result = idl_emit_udih(fout, parse);
    }
    else if (emit == EMIT_UDI_RECEIVER)
    {
        result = idl_emit_udirx(fout, parse);
    }
    else if (emit == EMIT_UDI_TRANSMITTER)
    {
        result = idl_emit_uditx(fout, parse);
    }
    else if (emit == EMIT_UDI_STANDARD)
    {
        result = idl_emit_udidef(fout, parse);
    }
    else
    {
        result = 1;
    }

    if (argc == 4)
        fclose(fout);

    idl_freeparse(parse);
    return result;
    
    
    return 0;
}
