
#include "idltool.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

IDL_LEX * idl_lexer(const char * input)
{
    int state = 0;
    while (state == 0)
    {
        char val = *input;
        if (val == 0)
        {
            state = 5;
        }
        else if (val == ' ' || val == '\t' || val == 0x0A || val == 0x0D)
        {                        
            input++;
        }
        else if (val == '#')
        {
            state = 2;
        }
        else if (val == '{')
        {        
            state = 3;
        }
        else if (val == '}')    
        {
            state = 4;
        }
        else if (val >= '0' && val <= '9')            
            state = 1;
        else if (val >= 'A' && val <= 'Z')            
            state = 1;
        else if (val >= 'a' && val <= 'z')
            state = 1;
        else if (val == '_')
            state = 1;
        else
        {        
            fprintf(stderr, "lexing error reading symbol %i (%c)\n", (int) *input, *input);
            return NULL;
        }
    }
    
    const char * dataoff = input;
    if (state == 1)
    {
        while (state == 1)
        {
            char val = *input;
            if (val >= '0' && val <= '9')
                input++;        
            else if (val >= 'a' && val <= 'z')
                input++;
            else if (val >= 'A' && val <= 'Z')
                input++;
            else if (val == '_')
                input++;
            else
                state = 0;            
        }
        char * content = malloc(input-dataoff+1);
        if (!content)
        {
            fprintf(stderr, "out of memory");
            return NULL;
        }
        IDL_LEX * lexeme = malloc(sizeof(IDL_LEX));
        if (!lexeme)
        {
            fprintf(stderr, "out of memory");
            free(content);
            return NULL;
        }
        
        memcpy(content, dataoff, input-dataoff);
        content[input-dataoff] = 0;
        lexeme->content = content;
        
        if (strcmp(content, "protocol") == 0)
            lexeme->type = TYPE_KEYWORD_PROTOCOL;
        else if (strcmp(content, "message") == 0)
            lexeme->type = TYPE_KEYWORD_MESSAGE;
        else if (strcmp(content, "uint8_t") == 0)
            lexeme->type = TYPE_KEYWORD_TYPE | DATA_8;
        else if (strcmp(content, "uint16_t") == 0)
            lexeme->type = TYPE_KEYWORD_TYPE | DATA_16;
        else if (strcmp(content, "uint32_t") == 0)
            lexeme->type = TYPE_KEYWORD_TYPE | DATA_32;
        else if (strcmp(content, "uint64_t") == 0)
            lexeme->type = TYPE_KEYWORD_TYPE | DATA_64;
        else if (strcmp(content, "mstring") == 0)
            lexeme->type = TYPE_KEYWORD_TYPE | DATA_STR;
        else if (strcmp(content, "struct") == 0)
            lexeme->type = TYPE_KEYWORD_STRUCT;
        else
            lexeme->type = TYPE_NAME;
        
        lexeme->next = idl_lexer(input);
        
        if (lexeme->next == NULL)
        {
            // TODO: clean up
            return NULL;
        }
        
        return lexeme;
        
    }
    else if (state == 2)
    {
        while (state == 2)
        {
            char val = *input;
            if (val == 0x0A || val == 0x0D || val == 0)        
                state = 0;            
            else
                input++;
        }
        
        char * content = malloc(input-dataoff+1);
        if (!content)
        {
            fprintf(stderr, "out of memory");
            return NULL;
        }
        IDL_LEX * lexeme = malloc(sizeof(IDL_LEX));
        if (!lexeme)
        {
            fprintf(stderr, "out of memory");
            free(content);
            return NULL;
        }
        
        memcpy(content, dataoff, input-dataoff);
        content[input-dataoff] = 0;
        lexeme->content = content;
        lexeme->type = TYPE_COMMENT;
        
        lexeme->next = idl_lexer(input);
        if (lexeme->next == NULL)
        {
            // TODO: clean up
            return NULL;
        }
        
        return lexeme;
    }
    else if (state == 3)
    {
        IDL_LEX * lexeme = malloc(sizeof(IDL_LEX));
        if (!lexeme)
        {
            fprintf(stderr, "out of memory");
        }
        lexeme->type = TYPE_BLOCKSTART;
        lexeme->content = NULL;
        
        lexeme->next = idl_lexer(input + 1);
        if (lexeme->next == NULL)
        {
            // TODO: clean up
            return NULL;
        }
        
        return lexeme;
    }
    else if (state == 4)
    {
        IDL_LEX * lexeme = malloc(sizeof(IDL_LEX));
        if (!lexeme)
        {
            fprintf(stderr, "out of memory");
        }
        lexeme->type = TYPE_BLOCKEND;
        lexeme->content = NULL;
        
        lexeme->next = idl_lexer(input + 1);
        if (lexeme->next == NULL)
        {
            // TODO: clean up
            return NULL;
        }
        
        return lexeme;
    }
    else if (state == 5)
    {
        IDL_LEX * lexeme = malloc(sizeof(IDL_LEX));
        if (!lexeme)
        {
            fprintf(stderr, "out of memory");
        }
        lexeme->type = TYPE_EOF;
        lexeme->content = NULL;        
        lexeme->next = NULL;        
        return lexeme;
    }
    else        
    {
        fprintf(stderr, "internal lexer error for state %u\n", state);
        return NULL;
    }
}

static const char * typestring(IDL_LEX * lex)
{
    switch (lex->type)
    {
        case 0:
	    return "end of file";
	case 1:
	    return "message keyword";
	case 2:
	    return "protocol keyword";
	case 4:
	    return lex->content;
	case 5:
	    return "comment";
	case 6:
	    return "{";
	case 7:
	    return "}";
	case 8:
	    return "struct";

        case 3:
        default:
	    return "typename";
    }
}

void idl_freelex(IDL_LEX * lex)
{
    IDL_LEX * next;
    while (lex)
    {
        next = lex->next;
        if (lex->content) free(lex->content);        
        free(lex);
        lex = next;
    }
}

IDL_CHANNEL * idl_parse_channel(IDL_LEX ** lexptr)
{
    char * srcname = NULL;
    IDL_LEX * lex = *lexptr;
    int ok = 1;

    //printf("%i Reading %p\n", __LINE__, lex);
    if (lex->type == TYPE_NAME)
    {
        srcname = lex->content;
        lex = lex->next;
    }
    else
    {
        *lexptr = lex;
        fprintf(stderr, "Parse error: expected channel name, found %s\n", typestring(lex));
        return NULL;
    }
    
    //printf("%i Reading %p\n", __LINE__, lex);
    if (lex->type == TYPE_BLOCKSTART)
    {
        lex = lex->next;
    }
    else
    {
        *lexptr = lex;
        fprintf(stderr, "Parse error: expected '{' after channel, found %s\n", typestring(lex));
        return NULL;
    }

    IDL_MESSAGE * head = NULL;
    IDL_MESSAGE * tail = NULL;
    
    while (lex->type == TYPE_KEYWORD_MESSAGE)
    {        
        //printf("%i Reading %p\n", __LINE__, lex);
        lex = lex->next;
        IDL_MESSAGE * message = idl_parse_message(&lex);
        *lexptr = lex;
        if (message == NULL)
        {	    
            ok = 0;
            //idl_freemessage(head);
            //return NULL;
        }
        else if (head == NULL)
        {
            head = message;
            tail = message;
        }
        else
        {
            tail->next = message;
            tail = message;
        }
    }

    //printf("%i Reading %p: %s\n", __LINE__, lex, typestring(lex));
    if (lex->type == TYPE_BLOCKEND)
    {
        lex = lex->next;
    }
    else
    {
        fprintf(stderr, "parse error: expected '}' at end of channel, found %s\n", typestring(lex));
        ok = 0;
    }
    //printf("%i asserting %p: %s\n", __LINE__, lex, typestring(lex));

    IDL_CHANNEL * channel = (IDL_CHANNEL *) malloc(sizeof(IDL_CHANNEL));
    char * copy = (char *) malloc(strlen(srcname)+1);
    if (copy == NULL || channel == NULL || ok == 0)
    {
        if (ok) fprintf(stderr, "out of memory");
        if (copy) free(copy);
        if (channel) free(channel);
        idl_freemessage(head);
        return NULL;
    }

    strcpy(copy, srcname);
    channel->interface_id = 0;
    channel->messages = head;
    channel->interface_name = copy;
    channel->next = NULL;
    
    *lexptr = lex;    
    return channel;
}

IDL_MESSAGE * idl_parse_message(IDL_LEX ** lexptr)
{
    char * srcname = NULL;
    IDL_LEX * lex = *lexptr;    

    //printf("%i Reading %p\n", __LINE__, lex);
    if (lex->type == TYPE_NAME)
    {
        srcname = lex->content;
        lex = lex->next;
    }
    else
    {
        *lexptr = lex;
        fprintf(stderr, "Parse error: expected name");
        return NULL;
    }

    if (lex->type == TYPE_BLOCKSTART)
    {
        lex = lex->next;
    }
    else
    {
        *lexptr = lex;
        fprintf(stderr, "Parse error: expected '{' after message name, found %s\n", typestring(lex));
        return NULL;
    }
           
    int ok = 0;
    IDL_PARAM * param = idl_parse_param(&lex, &ok);        

    *lexptr = lex;
    IDL_MESSAGE * message = (IDL_MESSAGE *) malloc(sizeof(IDL_MESSAGE));
    char * copy = (char *) malloc(strlen(srcname)+1);
    if (copy == NULL || message == NULL || ok == 0)
    {
        if (ok) fprintf(stderr, "out of memory");
        if (copy) free(copy);
        if (message) free(message);
        idl_freeparam(param);
        return NULL;
    }

    strcpy(copy, srcname);
    message->params = param;
    message->message_name = copy;
    message->next = NULL;

    return message;
}

IDL_PARAM * idl_parse_param(IDL_LEX ** lexptr, int * ok)
{
    IDL_PARAM * head = NULL;
    IDL_PARAM * tail = NULL;
    
    IDL_LEX * lex = *lexptr;
    *ok = 0;
    
    while ((lex->type & 0xff) == TYPE_KEYWORD_TYPE || lex->type == TYPE_KEYWORD_STRUCT )
    {   
        IDL_PARAM * param = NULL;
        
        if (lex->type == TYPE_KEYWORD_STRUCT)
        {
            lex = lex->next;
            if (lex->type != TYPE_NAME)
            {
                fprintf(stderr, "Parse error: expected typename name after struct type, found %s\n", typestring(lex));
                idl_freeparam(head);
                return NULL;
            }
            const char * typename = lex->content;
            
            lex = lex->next;
            if (lex->type != TYPE_NAME)
            {
                fprintf(stderr, "Parse error: expected second name after struct type, found %s\n", typestring(lex));
                idl_freeparam(head);
                return NULL;
            }
            const char * argname = lex->content;
            
            lex = lex->next;
            if (lex->type != TYPE_BLOCKSTART)
            {
                fprintf(stderr, "Parse error: expected { after struct name, found %s\n", typestring(lex));
                idl_freeparam(head);
                return NULL;
            }
            
            lex = lex->next;
            int childok = 0;
            IDL_PARAM * children = idl_parse_param(&lex, &childok);
            
            if (!childok)
            {
                fprintf(stderr, "in struct %s %s\n", typename, argname);
                idl_freeparam(head);
                return NULL;
            }
            
            // commit memory
            char * typecopy = malloc(strlen(typename)+1);
            char * namecopy = malloc(strlen(argname)+1);
            param = calloc(1, sizeof(IDL_PARAM));
            if (typecopy == NULL || namecopy == NULL || param == NULL)
            {
                fprintf(stderr, "Out of memory\n");
                idl_freeparam(head);
                if (param) free(param);
                if (typecopy) free(typecopy);
                if (namecopy) free(namecopy);
                return NULL;
            }
            
            // write data
            param->param_type = DATA_TUPLE >> 8;
            param->param_customtype = typecopy;
            param->param_name = namecopy;
            param->child = children;
            strcpy(typecopy, typename);
            strcpy(namecopy, argname);            
        }
        else
        {
            //printf("%i Reading %p\n", __LINE__, lex);
            int typename = lex->type >> 8;	
            lex = lex->next;

            //printf("%i Reading %p\n", __LINE__, lex);
            if (lex->type != TYPE_NAME)
            {
                fprintf(stderr, "Parse error: expected argument name after type, found %s\n", typestring(lex));
                idl_freeparam(head);
                return NULL;
            }

            param = calloc(1, sizeof(IDL_PARAM));
            char * name = malloc(strlen(lex->content)+1);
            if (param == NULL || name == NULL)
            {
                fprintf(stderr, "Out of memory\n");
                idl_freeparam(head);
                if (param) free(param);
                if (name) free(name);
                return NULL;
            }
            // parse argument
            param->param_type = typename;
            param->param_name = name;
            param->next = NULL;
            strcpy(name, lex->content);
            lex = lex->next;
                    
        }
        
        if (tail)
        {
            tail->next = param;
            tail = param;
        }
        else
        {
            head = param;
            tail = param;
        }

        *lexptr = lex;
    }
    
    //printf("%i Reading %p\n", __LINE__, lex);
    if (lex->type == TYPE_BLOCKEND)
    {
        lex = lex->next;
	*lexptr = lex;
	*ok = 1;
	return head;
    }
    else
    {
        *lexptr = lex;
        fprintf(stderr, "Parse error: expected param or '}', found %s\n", typestring(lex));
	if (head != NULL) idl_freeparam(head);
        return NULL;
    }
}

IDL_CHANNEL * idl_parser(IDL_LEX * lex)
{
    IDL_CHANNEL * head = NULL;
    IDL_CHANNEL * tail = NULL;
    int ok = 1;
    
    while (lex)
    {
        //printf("%i Reading %p\n", __LINE__, lex);
        if (lex->type == TYPE_KEYWORD_PROTOCOL)
        {
            lex = (IDL_LEX *) lex->next;	    
            IDL_CHANNEL * child = idl_parse_channel(&lex);	    
            if (!child)
            {
                ok = 0;
            }
            else if (tail)
            {
                tail->next = child;
                tail = child;
            }
            else
            {
                tail = head = child;
            }
        }
        else if (lex->type == TYPE_EOF)
        {
            lex = lex->next;
            if (ok) return head;
            idl_freeparse(head);
            return NULL;
        }
        else
        {
            ok = 0;
            fprintf(stderr, "parse error: Expected keyword protocol, found %s\n", typestring(lex));
            lex = lex->next;
        }
    }        
    
    fprintf(stderr, "parse error: Unexpected end of file\n");
    idl_freeparse(head);
    return NULL;
}

void idl_freeparse(IDL_CHANNEL * parse)
{
    while (parse != NULL)
    {
	IDL_CHANNEL * next = (IDL_CHANNEL *) parse->next;
        if (parse->interface_name != NULL) free(parse->interface_name);
	idl_freemessage(parse->messages);
	free(parse);
	parse = next;
    }
}

void idl_freemessage(IDL_MESSAGE * message)
{
    while (message != NULL)
    {
        IDL_MESSAGE * next = (IDL_MESSAGE *) message->next;
	if (message->message_name != NULL) free(message->message_name);
        idl_freeparam(message->params);
	free(message);
	message = next;
    }
}

void idl_freeparam(IDL_PARAM * param)
{
    while (param != NULL)
    {
        IDL_PARAM * next = (IDL_PARAM *)param->next;
        if (param->param_name != NULL) free(param->param_name);
	if (param->param_customtype != NULL) free(param->param_customtype);
	if (param->child != NULL) idl_freeparam(param->child);
        free(param);
        param = next;
    }
}

