' Summary: atactl.bas
' The driver for a generic ATA controller.
'
' Author:
'     Marcel Sondaar
'
' License:
'     <Public Domain>
'

#include "mos/drivercom.bi"
#include "mos/driver.bi"
#include "mos/pci.bi"
#include "mos/atabus.bi"
#include "mos/devmgr.bi"
#include "mos.bi"
#include "x86.bi"
#include "atadef.bi"

Type ATABUS
    baseport As Integer
    extendedport As Integer
    dmaport As Integer
End Type

Type ATADRIVE
    bus As ATABUS
    sysdevice As Integer
    isprimary As Byte
    ispacket As Byte
    atalevel As Byte
    ispresent As Byte
    senddata As Byte Ptr
    replyaddr As Integer
    isbusy As Byte
End Type


Dim shared vram As Byte Ptr

Sub showbyte(ByVal b As Integer)
    outportb(&HE9, b)

    If vram = Cptr(Byte Ptr, 0) Then Exit Sub

    vram[0] = b And &HFF&
    vram[1] = &H1F
    vram[2] = 0
    vram[3] = &H7F
    vram = @(vram[2])
End Sub

Sub showstring(ByRef s As String)
    Dim lp As Integer
    For lp = 1 to len(s) 
        showbyte(asc(mid$(s, lp, 1)))
    Next lp
End Sub

Declare Sub ATA_Write_PIO(ByVal drive As ATADRIVE Ptr, ByVal buffer as Byte Ptr, ByVal length As Integer)
Declare Sub ATA_Read_PIO(ByVal drive As ATADRIVE Ptr, ByVal buffer as Byte Ptr, ByVal length As Integer)
Declare Sub ATA_ProbeDevice(ByVal drive As ATADRIVE Ptr)
Declare Sub ATA_PollDevices(ByVal drive As ATADRIVE Ptr)
Declare Sub ATA_WriteCommand(ByVal drive As ATADRIVE Ptr, ByVal buffer As Byte Ptr, ByVal buflen As Integer)
Declare Sub ATA_AddDrive(ByVal drive As ATADRIVE Ptr, ByVal parent As Integer)

Dim Shared devices(0 to 3) as ATADRIVE

Sub modmain CDecl Alias "main" (ByVal argc As Integer, ByVal argv As Byte Ptr Ptr)

    allocateiobitmap(0, &HE000, CPtr(Byte Ptr, &HFFFFFFFF))
    portalloc(&HE9, 1)
    portalloc(ATA_ISABASE_1, 16)
    portalloc(ATA_ISABASE_2, 16)

    Dim devid As Integer
    devid = DriverInit

    Dim primary as ATABUS
    Dim secondary as ATABUS

    primary.baseport = ATA_ISABASE_1
    secondary.baseport = ATA_ISABASE_2

    ' register driver in device manager
    drv_setname(0,1)
    Dim msgsize As Integer
    Dim msg() As integer
    Redim msg(2)
    msg(0) = DEVMGRCOMMANDS_SETADDR
    msg(1) = devid
    drv_sendmessage(DRIVER_MGR * &H10000 + 0, 8, CPtr(Byte Ptr, @(msg(0))) )

    ' start initial enumeration
    devices(0).bus = primary
    devices(1).bus = primary
    devices(2).bus = secondary
    devices(3).bus = secondary
    Dim lp As Integer
    For lp = 0 To 3
        devices(lp).ispresent = 0
        devices(lp).ispacket = 0
        devices(lp).atalevel = 0
        devices(lp).isprimary = 1 - (lp And 1)
        devices(lp).isbusy = 0
        ATA_probedevice(@(devices(lp)))

        If devices(lp).ispresent Then ATA_AddDrive(@(devices(lp)), devid)
    Next lp

    ' main message loop
    While 1 = 1
        msgsize = drv_PeekMessage()
        If msgsize = 0 Then
            Yield
            Dim lp As Integer
            For lp = 0 To 3
                ATA_PollDevices(@devices(lp))
            Next lp
        Else
            'to enable debug output to screen
            'If vram = Cptr(Byte Ptr, 0) Then
                'vram = CPtr(Byte Ptr, &HB8000)
                'blockallocphys(8, vram, vram)
            'End If
        
            Dim src As Integer
            ReDim msg(msgsize \ 4 + 1)
            src = drv_readmessage(CPtr(Byte Ptr, @(msg(0)) )  )

            Select Case msg(0)
                Case SYSCOMMAND_QUERYINTERFACES
                    Showstring "[Query]"
                    Redim msg(2)
                    msg(0) = SYSCOMMAND_QUERYINTERFACES
                    msg(1) = INTERFACE_ATABUS
                    drv_sendmessage(src, 8, CPtr(Byte Ptr, @(msg(0))))

                Case ATABUSCOMMAND_CMD
                    Dim lp As Integer, bits As Integer
                    Showstring "[Command]"
                    For lp = 0 To 3
                        showbyte(asc("U"))
                        If (devices(lp).sysdevice = msg(1)) And (msg(1) <> 0) Then
                            showbyte(asc("V"))
                            While devices(lp).isbusy
                                showbyte(asc("W"))
                                ATA_PollDevices(@devices(lp)) ' complete previous command first
                                Yield
                            Wend
                            showbyte(asc("X"))
                            devices(lp).replyaddr = src
                            showbyte(asc("Y"))
                            showbyte(asc("0") + msgsize - 9)
                            ATA_WriteCommand(@devices(lp), @(Cptr(Byte Ptr, @(msg(0)))[9]), msgsize - 9)
                            showbyte(asc("Z"))
                        End If
                    Next lp
                    Showstring "[/Command]"
                Case Else
                    *CPtr(Byte Ptr, &H61700000 + msg(0) ) = 0
            End Select
        End If
    Wend

End Sub

Sub ATA_Read_PIO(ByVal drive As ATADRIVE Ptr, ByVal buffer as Byte Ptr, ByVal length As Integer)
    Dim port As integer
    Showstring "[PIO Read]"
    port = drive[0].bus.baseport
    Asm
        mov ecx, [length]
        mov dx, [port]
        shr ecx, 1
        mov edi, [buffer]
        cld
        rep insw
    End Asm
End Sub


Sub ATA_Write_PIO(ByVal drive As ATADRIVE Ptr, ByVal buffer as Byte Ptr, ByVal length As Integer)
    Showstring "[PIO Write]"
    Dim port As integer
    port = drive[0].bus.baseport
    Asm
        mov ecx, [length]
        mov dx, [port]
        shr ecx, 1
        mov esi, [buffer]
        cld
        rep outsw
    End Asm
End Sub

Sub ATA_ProbeDevice(ByVal drive As ATADRIVE Ptr)
    Showstring "[Probe] "
    Dim port As integer
    port = drive[0].bus.baseport
    drive[0].ispresent = 0

    If inportb(port+ATA_R_STATUS) = &HFF Then Exit Sub ' floating bus = no devices attached
    outportb(port+ATA_RW_DEVSEL, iif(drive[0].isprimary = 1, &HA0, &HB0))

    showbyte(asc("1"))

    Dim lp As Integer
    Dim rb as byte
    For lp = 0 to 4
        rb = inportb(port+ATA_R_STATUS)
    Next lp

    showbyte(asc("2"))

    While (rb And &HC0) <> ATA_STATUS_DRDY
        If (rb And &HC0) = &HC0 or (rb And &HC0) = &H00 Then Exit Sub ' invalid statuses
        If (rb And (ATA_STATUS_DF Or ATA_STATUS_ERR)) > 0 Then Exit Sub ' Drive Fault or Error
        Yield
        rb = inportb(port+ATA_R_STATUS)
        'showbyte(rb)
    Wend

    'showbyte(asc("3"))

    outportb(port+ATA_W_COMMAND, ATACOMMAND_IDENTIFY) ' Identify command

    'showbyte(asc("4"))

    rb = inportb(port+ATA_R_STATUS)
    While (rb And (ATA_STATUS_BUSY Or ATA_STATUS_DRQ)) <> ATA_STATUS_DRQ ' until BSY clear, DRQ set
        If rb = 0 Then Exit Sub ' does not exists
        If (rb and ATA_STATUS_ERR) = ATA_STATUS_ERR Then
            ' Todo: ATAPI/SATA checking
            'showbyte(asc("!"))
            Showstring "[/Probe]"
            Exit Sub ' device error or packet device
        End If
        If (rb And ATA_STATUS_DF) = ATA_STATUS_DF Then Exit Sub ' Drive Fault
	Yield
	rb = inportb(port+ATA_R_STATUS)
    Wend

    'showbyte(asc("5"))

    Dim buffer() As Byte
    Redim buffer(512)

    ATA_Read_PIO(drive, @(buffer(0)), 512)

    'showbyte(asc("6"))

    Dim ataident as ATAIDENTIFY Ptr
    ataident = CPtr(ATAIDENTIFY Ptr, @(buffer(0)))

    drive[0].ispresent = 1
    Dim bits As Unsigned Short
    bits = ataident->versionsupport
    drive[0].atalevel = 0
    while bits > 0
        drive[0].atalevel = drive[0].atalevel + 1
        bits = bits SHR 1
    wend
    Showstring "[/Probe]"
    'drive[0].atalevel = ataident->versionsupport And &HFF
End Sub

Sub ATA_AddDrive(ByVal drive As ATADRIVE Ptr, ByVal parent As Integer)
    Showstring "[AddDrive]"
    Dim message(0 to 1) As Integer
    message(0) = DEVMGRCOMMANDS_ADD
    message(1) = parent
    drv_sendmessage(DRIVER_MGR * &H10000, 8, CPtr(Byte Ptr, @(message(0)) )  )

    While drv_peekmessage() = 0
        Yield
    Wend

    drv_readmessage(CPtr(Byte Ptr, @(message(0)) ) )
    drive[0].sysdevice = message(0)

    ' set device name
    Dim ws As String, nodename As String
    If drive[0].ispacket = 1 Then
        nodename = "ATAPI-"
    Else
        nodename = "ATA-"
    End If
    ws = "        " & nodename & drive[0].atalevel
    Dim bp As Byte Ptr
    bp = *Cptr(Byte Ptr Ptr, @ws)
    Cptr(Integer Ptr, bp)[0] = DEVMGRCOMMANDS_SETNAME
    Cptr(Integer Ptr, bp)[1] = message(0)
    drv_sendmessage(DRIVER_MGR * &H10000, len(ws), bp)

    ' set device location
    ws = "        " & "ATA " & Str$(2 - drive[0].isprimary)
    bp = *Cptr(Byte Ptr Ptr, @ws)
    Cptr(Integer Ptr, bp)[0] = DEVMGRCOMMANDS_SETLOCATION
    Cptr(Integer Ptr, bp)[1] = message(0)
    drv_sendmessage(DRIVER_MGR * &H10000, len(ws), bp)

    Showstring "[/AddDrive]"
End Sub

Sub ATA_PollDevices(ByVal drive As ATADRIVE Ptr)
    If drive[0].isbusy = 0 Then Exit Sub
    
    showbyte(asc("."))

    Dim port As Integer
    port = drive[0].bus.baseport
    outportb(port + ATA_RW_DEVSEL, iif(drive[0].isprimary = 1, &HA0, &HB0))

    Dim rb As Byte, lp As Integer
    For lp = 0 to 4
        rb = inportb(port + ATA_R_STATUS)
    Next lp
    If (rb And ATA_STATUS_ERR) <> 0 Or (rb And ATA_STATUS_DF) <> 0 Then
        ' hardware error
        ' TODO: send packet
        showbyte(Asc("F"))
        drive[0].isbusy = 0
    End If
    If (rb And ATA_STATUS_DRQ) > 0 Then
        ' The device expects a data transfer in one direction or the other.
        showstring("(drq)")
        If (drive[0].isbusy And 4) = 4 Then
            If drive[0].senddata = CPtr(Byte Ptr, 0) Then Exit Sub ' no data packet received yet
            ATA_Write_PIO(drive, drive[0].senddata, 512)
            drive[0].isbusy = drive[0].isbusy And &HFB
        ElseIf (drive[0].isbusy And 2) = 2 Then
            Dim m() as Byte
            ReDim m(516)
            *Cptr(Integer Ptr, @(m(0))) = ATABUSCOMMAND_DATA
            ATA_Read_PIO(drive, CPtr(Byte Ptr, @(m(4))), 512)
            drv_sendmessage(drive[0].replyaddr, 516, CPtr(Byte Ptr, @(m(0))))
            Yield
        Else
            ' software error
            ' TODO: send packet
            showbyte(Asc("f"))
            drive[0].isbusy = 0
        End If
    ElseIf (rb And ATA_STATUS_BUSY) > 0 Then
        ' Needs to be checked here since real hardware can and does lie about the drdy bit
        ' and have it set when other conflicting ones are set too. 
        Showstring "*"
    ElseIf (rb And ATA_STATUS_DRDY) > 0 Then
        Showstring "(drdy)"
        ' TODO: send queued packet
        drive[0].isbusy = 0
    End If

End Sub

Sub ATA_WriteCommand(ByVal drive As ATADRIVE Ptr, ByVal buffer As Byte Ptr, ByVal buflen As Integer)
    Showstring "[WriteCMD]"
    drive[0].isbusy = (buffer[0] SHL 1) + 1
    
    Dim port As Integer
    port = drive[0].bus.baseport
    outportb(port + ATA_RW_DEVSEL, iif(drive[0].isprimary = 1, &HA0, &HB0))

    Dim lp As Integer, bits As Integer, bitoff As Integer
    For lp = 0 to 3
        inportb(port + ATA_R_STATUS)
    Next lp

    showbyte(asc("<"))

    lp = 1
    While lp < buflen
        bits = 0
        If lp + 1 < buflen Then
            bits = buffer[lp] + (buffer[lp+1] SHL 8)
            showbyte(asc("*"))
            showbyte(buffer[lp])
            showbyte(buffer[lp+1])
            showbyte(asc("-"))
            showbyte(asc("0") + buffer[lp])
            showbyte(asc("0") + buffer[lp+1])
            lp = lp + 2
        Else
            showbyte(asc("?"))
            lp = lp + 1
        End If
        bitoff = 0
        While lp < buflen And bits <> 0
            If (bits And (1 SHL bitoff)) <> 0 Then
                bits = bits And (not (1 SHL bitoff))
                Select Case bitoff
                    Case 0 To 1, 8 to 15
                        showbyte(asc("|"))
                        showbyte(asc("0") + bitoff)
                        ' never write cases

                    Case 2 To 5
                        ' always write cases
                        showbyte(asc(":"))
                        showbyte(buffer[lp])
                        outportb(port + bitoff, buffer[lp])

                    Case 6
                        showbyte(asc("|"))
                        showbyte(asc("0") + bitoff)
                        outportb(port + bitoff, (buffer[lp] And &HEF) Or iif(drive[0].isprimary = 1, &H00, &H10))
                        ' devsel, mask bits

                    Case 7
                        ' command register
                        showbyte(asc("#"))
                        showbyte(buffer[lp])
                        outportb(port + bitoff, buffer[lp])
                        lp = buflen ' always the last command
                        bits = 0
                    Case Else
                        showbyte(asc("?"))
                        showbyte(asc("0") + bitoff)
                End Select
                lp = lp + 1
            End If
            bitoff = bitoff + 1
        Wend
    Wend
    showbyte(asc(">"))
    Showstring "[/WriteCMD]"

End Sub
