' Summary: diskmm.bas
' Support code for disk access management
'
' Author:
'     Marcel Sondaar
'
' License:
'     <Public Domain>
'

#include "diskmm.bi"

Dim Shared Lastmapping As Integer
Dim Shared Freemappings As Integer
Dim Shared mappings() As DISKMM_MAPPING Ptr
Dim Shared mapstack() As Integer

Sub DiskMM_InitMapper()
    Lastmapping = 1
    ReDim mappings(1)
    ReDim mapstack(1)
    mapstack(0) = 0
    mappings(0) = CPtr(DISKMM_MAPPING Ptr, 0)
    Freemappings = 1
End Sub

Function DiskMM_NewMapping() As Integer

    ' Try to get a free slot
    If Freemappings > 0 Then
        Dim map As Integer
        Freemappings = Freemappings - 1
        map = mapstack(Freemappings)
        mappings(map) = DiskMM_CreateMapping()

        ' handle memory errors
        If mappings(map) = CPtr(DISKMM_MAPPING Ptr, 0) Then
            Freemappings = Freemappings + 1
            Function = -1
            Exit Function
        End If

        Function = map
        Exit Function
    End If

    ' Increase the number of maps allowed
    Dim newmappings As Integer
    newmappings = lastmapping * 2 ' store the new end position
    ReDim Preserve mappings(newmappings)
    ReDim Preserve mapstack(newmappings)

    Dim lp As Integer
    For lp = Lastmapping to newmappings - 1
        mapstack(freemappings) = lp
        freemappings = freemappings + 1
    Next lp
    lastmapping = newmappings

    ' Try again and save some code duplication
    Function = DiskMM_NewMapping()
End Function

Function DiskMM_CreateMapping() As DISKMM_MAPPING Ptr
    Dim mapping As DISKMM_MAPPING Ptr
    mapping = CPtr(DISKMM_MAPPING Ptr, Allocate(sizeof(DISKMM_MAPPING)))
    Function = mapping
    If mapping = CPtr(DISKMM_MAPPING Ptr, 0) Then Exit Function

    mapping->size = 0
    mapping->root = CPtr(DISKMM_EXTENTTREE Ptr, 0)
    mapping->privileges = 0
End Function

Sub DiskMM_SetMapProperties(ByVal Map as Integer, ByVal Owner as Integer, ByVal Size as Long)
    mappings(map)->Owner = Owner
    mappings(map)->Size = Size
    mappings(map)->Privileges = SECTORSTATUS_RW
End Sub

Function DiskMM_FillSingle(ByVal map As DISKMM_MAPPING Ptr, ByVal Owner as Integer, ByVal Size as Long) As Integer
    map->Owner = Owner
    map->Size = Size
    map->Privileges = SECTORSTATUS_RW
    map->Root = CPtr(DISKMM_EXTENTTREE Ptr, Allocate(sizeof(DISKMM_EXTENTTREE)))
    If map->Root = CPtr(DISKMM_EXTENTTREE Ptr, 0) Then
        Function = 1
        Exit Function
    End If
    map->Root->Extent.StartLocation = 0
    map->Root->Extent.EndLocation = Size - 1
    map->Root->Extent.ParentLevel = 0
    map->Root->Extent.ParentStart = 0
    map->Root->Extent.Privileges = SECTORSTATUS_RW
    map->Root->Left = CPtr(Byte Ptr, 0)
    map->Root->Right = CPtr(Byte Ptr, 0)
    map->Root->Weight = 1
End Function



Sub DiskMM_CreateRoot(ByVal Owner As Integer, ByVal Size as Long)
    Dim map As Integer
    map = DiskMM_NewMapping
    If map <> 0 Then *cptr(Byte Ptr, &HED170001) = 0
    DiskMM_FillSingle(mappings(map), Owner, Size)
End Sub

Function DiskMM_Translate(ByVal map As Integer, ByVal Requester as Integer, ByRef Extent As DISKMM_EXTENT) As DISKMM_TRANSLATION Ptr
    ' sanitize input
    If map < 0 Or map >= Lastmapping Then
        Function = CPtr(DISKMM_TRANSLATION Ptr, 0)
        Exit Function
    End If
    If mappings(map) = CPtr(DISKMM_MAPPING Ptr, 0) Then
        Function = CPtr(DISKMM_TRANSLATION Ptr, 0)
        Exit Function
    End If

    Dim unused As Integer

    ' proceed with actual translation
    Function = DiskMM_TranslateUnsafe(Map, Requester, @Extent, Unused)

End Function

Function DiskMM_TranslateUnsafe(ByVal map As Integer, ByVal Requester as Integer, ByVal Extent As DISKMM_EXTENT Ptr, Byref Validated as Integer) As DISKMM_TRANSLATION Ptr
    Dim mapping As DISKMM_MAPPING Ptr
    mapping = mappings(map)

    Dim TranslationTail As DISKMM_TRANSLATION Ptr
    TranslationTail = CPtr(DISKMM_TRANSLATION Ptr, 0)

    Dim myextent As DISKMM_EXTENT
    myextent = *Extent

    If Extent->EndLocation > mapping->size Then
        Dim Postfail As DISKMM_TRANSLATION Ptr
        Postfail = CPtr(DISKMM_TRANSLATION Ptr, Allocate(sizeof(DISKMM_TRANSLATION)))
        Postfail->OriginalStart = mapping->size + 1
        Postfail->NextBlock = CPtr(Byte Ptr, TranslationTail)
        Postfail->Extent.Startlocation = 0
        Postfail->Extent.Endlocation = Extent->Endlocation - mapping->size
        Postfail->Extent.Privileges = 0
        Postfail->Extent.Parentlevel = 0
        myextent.EndLocation = mapping->size
        TranslationTail = Postfail
    End If

    If Extent->Startlocation < 0 Then myextent.Startlocation = 0

    If mapping->Owner = Requester Then validated = 1

    Dim midpart As DISKMM_TRANSLATION Ptr
    midpart = DiskMM_FoldTranslate(mapping->Root, Requester, @myextent, validated)

    ' Todo: concatenate fail to end if necessary
    translationtail = midpart

    If Extent->StartLocation < 0 Then
        Dim Prefail As DISKMM_TRANSLATION Ptr
        Prefail = CPtr(DISKMM_TRANSLATION Ptr, Allocate(sizeof(DISKMM_TRANSLATION)))
        Prefail->OriginalStart = Extent->Startlocation
        Prefail->NextBlock = CPtr(Byte Ptr, TranslationTail)
        Prefail->Extent.Startlocation = 0
        Prefail->Extent.Endlocation = -1 - Extent->Startlocation
        Prefail->Extent.Privileges = 0
        Prefail->Extent.Parentlevel = 0
        TranslationTail = Prefail
    End If

    Function = TranslationTail
End Function



' Function: DiskMM_FoldTranslate
' Translates an extent for a given mapping to a list of extents in the physical domain.
'
' in:
'     node - pointer to the root node of an extent tree to translate
'     requester - the source making the request, used in determining validation
'     extent - pointer to the input extent
'     validated - nonzero if ownership was already established
'
' out:
'     return - a linked list of translated extents
'
Function DiskMM_FoldTranslate(ByVal node As DISKMM_EXTENTTREE Ptr, ByVal Requester As Integer, ByVal Extent as DISKMM_EXTENT Ptr, ByVal Validated As Integer) As DISKMM_TRANSLATION Ptr
    Dim functiontmp As DISKMM_TRANSLATION Ptr

    ' Check for leaf
    If node = CPtr(DISKMM_EXTENTTREE Ptr, 0) Then
        Dim fail As DISKMM_TRANSLATION Ptr
        fail = CPtr(DISKMM_TRANSLATION Ptr, Allocate(sizeof(DISKMM_TRANSLATION)))
        fail->Extent = *Extent
        fail->Extent.Privileges = 0
        fail->Extent.ParentLevel = 0
        fail->Extent.ParentStart = 0
        Function = Fail
        Exit Function
    End If

    ' check for everything to the right
    If Extent->Startlocation > node->Extent.EndLocation Then
        Function = DiskMM_FoldTranslate(CPtr(DISKMM_EXTENTTREE Ptr, node->Right), Requester, Extent, Validated)
        Exit Function
    End If
    ' or everything to the right
    If Extent->Endlocation < node->Extent.StartLocation Then
        Function = DiskMM_FoldTranslate(CPtr(DISKMM_EXTENTTREE Ptr, node->Left), Requester, Extent, Validated)
        Exit Function
    End If

    functiontmp = CPTR(DISKMM_TRANSLATION Ptr, 0) ' current head of linked list

    ' Start filling in, first right child so the output stays sorted
    If Extent->Endlocation > node->Extent.Endlocation Then

        Dim RightExtent As DISKMM_EXTENT
        RightExtent.StartLocation = node->Extent.Endlocation + 1
        RightExtent.EndLocation = Extent->Endlocation
        Dim rnode As DISKMM_TRANSLATION Ptr
        rnode = DiskMM_FoldTranslate(CPtr(DISKMM_EXTENTTREE Ptr, node->Right), Requester, @RightExtent, Validated)
        If rnode <> CPtr(DISKMM_TRANSLATION Ptr, 0) Then
            rnode->Nextblock = CPtr(Byte Ptr, functiontmp)
        End If
        functiontmp = rnode
    End If

    ' Everything to do with the current node.
    Dim midextent as DISKMM_EXTENT
    midextent = *Extent
    If Extent->Endlocation > node->Extent.Endlocation Then midextent.Endlocation = Extent->Endlocation
    If Extent->Startlocation < node->Extent.Startlocation Then midextent.Startlocation = Extent->Startlocation
    If node->Extent.ParentLevel = 0 Then
        Dim midfold As DISKMM_TRANSLATION Ptr
        midfold = CPtr(DISKMM_TRANSLATION Ptr, Allocate(sizeof(DISKMM_TRANSLATION)))
        midfold->Extent = midextent
        midfold->Extent.StartLocation = midfold->Extent.StartLocation + node->Extent.ParentStart
        midfold->Extent.EndLocation = midfold->Extent.EndLocation + node->Extent.ParentStart
        midfold->Nextblock = CPtr(Byte Ptr, functiontmp)
        If Validated = 0 Then midfold->Extent.Privileges = 0
        functiontmp = midfold
    Else
        *Cptr(byte ptr, &H0ED50006) = 0
        ' Todo: Recurse
    End If



    ' And check if the left child is needed
    If Extent->StartLocation < node->Extent.StartLocation Then
        Dim RightExtent As DISKMM_EXTENT
        RightExtent.StartLocation = Extent->StartLocation
        RightExtent.EndLocation = node->Extent.StartLocation - 1
        Dim lnode As DISKMM_TRANSLATION Ptr
        lnode = DiskMM_FoldTranslate(CPtr(DISKMM_EXTENTTREE Ptr, node->Right), Requester, @RightExtent, Validated)
        If lnode <> CPtr(DISKMM_TRANSLATION Ptr, 0) Then
            lnode->Nextblock = CPtr(Byte Ptr, functiontmp)
        End If
        functiontmp = lnode
    End If

    Function = functiontmp

End Function



' Function: DiskMM_Insert
' Inserts a translation extent into a map
'
' in:
'     map    - the map index to update
'     extent - the extent to insert
'
' out
'     return - zero on success, nonzero on failure
'
' failures include non-present maps, extents overlapping existing 
' extents, and extents (partially) falling outside of the map's range
'
Function DiskMM_Insert(ByVal map As Integer, ByRef extent as DISKMM_EXTENT) As Integer

    Function = 1

    ' Sanitize the map argument
    If map < 0 Or map >= Lastmapping Then Exit Function
    If mappings(map) = CPtr(DISKMM_MAPPING Ptr, 0) Then Exit Function

    Dim entry As DISKMM_EXTENTTREE Ptr
    entry = CPtr( DISKMM_EXTENTTREE Ptr, Allocate(sizeof(DISKMM_EXTENTTREE)))
    If entry = CPtr(DISKMM_EXTENTTREE Ptr, 0) Then Exit Function

    entry->extent = extent

    Dim retval As Integer
    retval = DiskMM_InsertNodeEx(mappings(map), entry)
    If retval = 1 Then
        Deallocate(entry)
        Exit Function
    End if

    ' success
    Function = 0
End Function



' Function: DiskMM_InsertNodeExPrime
' Insert an extent into a mappings extent tree.
' This function will not update anything if the inserted node overlaps existing nodes.
'
' in:
'     root    - current root of the tree, must be nonzero
'     newnode - new node to insert, must be nonzero
'
' out:
'     return  - zero on success, nonzero if there was any overlap
'
Function DiskMM_InsertNodeExPrime(ByVal root As DISKMM_EXTENTTREE Ptr, ByVal newnode As DISKMM_EXTENTTREE Ptr) As Integer
    If newnode->extent.StartLocation <= root->extent.EndLocation And newnode->extent.EndLocation >= root->extent.StartLocation Then
        ' Overlap, fail
        Function = 1
        Exit Function
    End If

    Dim retval As Integer
    If newnode->extent.StartLocation < root->extent.Startlocation Then
        ' left insert
        If root->Left = CPtr(Byte Ptr, 0) Then
            root->Left = CPtr(Byte Ptr, newnode)
            newnode->Weight = 1
            If root->Weight < 2 Then root->Weight = 2
        Else
            retval = DiskMM_InsertNodeExPrime(CPtr(DISKMM_EXTENTTREE Ptr, root->left), newnode)
            If retval > 0 Then
                Function = retval
                Exit Function
            End If
            if root->Weight <= CPtr(DISKMM_EXTENTTREE Ptr, root->left)->weight Then root->Weight = root->Weight + 1
        End If
    Else
        ' right insert
        If root->Right = CPtr(Byte Ptr, 0) Then
            root->Right = CPtr(Byte Ptr, newnode)
            newnode->Weight = 1
            If root->Weight < 2 Then root->Weight = 2
        Else
            retval = DiskMM_InsertNodeExPrime(CPtr(DISKMM_EXTENTTREE Ptr, root->Right), newnode)
            If retval > 0 Then
                Function = retval
                Exit Function
            End If
            if root->Weight <= CPtr(DISKMM_EXTENTTREE Ptr, root->Right)->weight Then root->Weight = root->Weight + 1
        End If
    End If

    ' Todo: Balance

End Function



' Function: DiskMM_InsertNodeEx
' Insert an extent into a mappings extent tree.
' This function will not update anything if the inserted node overlaps existing nodes.
'
' in:
'     map     - map to modify
'     newnode - new node to insert
'
' out:
'     return  - zero on success, nonzero if there was any overlap, any null argument, or if the extent is out of bounds
'
Function DiskMM_InsertNodeEx(ByVal map As DISKMM_MAPPING Ptr, ByVal newnode As DISKMM_EXTENTTREE Ptr) As Integer
    Function = 1
    ' Sanitize
    If map = CPtr(DISKMM_MAPPING Ptr, 0) Then Exit Function
    If newnode = CPtr(DISKMM_EXTENTTREE Ptr, 0) Then Exit Function
    If newnode->extent.Startlocation < 0 Then Exit Function
    If newnode->extent.Endlocation > map->size Then Exit Function

    If map->Root = CPtr(DISKMM_EXTENTTREE Ptr, 0) Then
        ' tree was empty
        Function = 0
        map->Root = newnode
        newnode->weight = 1
    Else
        ' tree insert
        Function = DiskMM_InsertNodeExPrime(map->Root, newnode)
    End If

End Function
