' Summary: test_ati.bas
' Testing module for ATI graphics stuff
'
' Author:
'     Marcel Sondaar
'
' License:
'     Educational Purposes
'

'$include once: 'vga_io.bi'
'$include once: 'mos.bi'
'$include once: 'x86.bi'
'$include once: 'mos/pci.bi'

Declare Sub ModMain Cdecl Alias "main" ()
Declare Sub PrintString (s As String, vram As Byte Ptr, offset As Integer)
Declare Function GetYesNo () As Integer
Declare Function ReadKB () as Integer
Declare Sub ClearCon (vram As Byte Ptr)
Declare Function Verite_PCIProbe(ByRef base_lfb As Long, ByRef base_pio As Integer) As Long

Declare Sub WriteLfbString(lfb As Byte Ptr, s as String, offset as Long)
Declare Sub PutPixel(vram As Byte Ptr, x As Long, y As Long, col As Long, bpp As Byte, pitch As Long)
Declare Sub WriteGfxString(vram as Byte Ptr, s as String, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
Declare Sub WriteGfxChar(vram as Byte Ptr, ch as Integer, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
Declare Sub WriteConioString(vram As Byte Ptr, s as String, font() As Byte, row As Long, column As Long, col As Long, bpp as Byte, vw As Long)
Declare Sub RenderTest(vram as Byte Ptr, w as long, h as long, bpp as byte, vw as long)

Declare Sub TestVerite(gpu_model As Long, gpu_version As Long, lfb As Long, portbase As Long)

' Enumeration: VREGS
' Registers common to all Verite-based boards
Enum VREGS

    V1K_FIFO = &H00

    ' Constant: VERITE_MEMENDIAN
    ' This register controls the byteswapping performed when
    ' writing to the linear framebuffer.
    '
    ' valid register values:
    ' 0 - no byteswapping
    ' 1 - 32-bit byteswapping
    ' 2 - 16-bit byteswapping
    ' 3 - swap halfwords
    '
    VERITE_MEMENDIAN = &H43
    
    ' Constant: VERITE_GPUCTL
    ' This register controls the state of the RISC processor
    ' It holds several bitflags:
    ' 01h - reset chip
    ' 02h - hold risc
    ' 04h - single step
    ' 08h - divide by 2 disable
    ' 10h - VGA reset
    ' 20h - assert XReset output to ext devices
    VERITE_GPUCTL = &H48
    
    ' Constant: VERITE_MODE
    ' This register allows one to select between CRTC modes.
    ' One can choose between VGA, VESA and Native mode by
    ' encoding the proper bits
    '
    ' Bitflags:
    '  01h - Decode 0xA0000.
    '  02h - Operate in VGA Mode. This will enable A0000-BFFFF 
    '        indepent of bit 1
    '  04h - 32-bit VGA
    VERITE_MODE = &H72
    
    ' Constant: VERITE_CRTCCTL
    ' CRTC Control register. 32-bits in size, contains several flags
    ' and pixel formats
    '
    ' Bitflags:
    '  0000000F - color format
    '  00000010 - CRTCCTL_VIDEOFIFOSIZE128
    '  00000020 - CRTCCTL_ENABLEDDC
    '  00000040 - CRTCCTL_DDCOUTPUT
    '  00000080 - CRTCCTL_DDCDATA
    '  00000100 - CRTCCTL_VSYNCHI
    '  00000200 - CRTCCTL_HSYNCHI
    '  00000400 - CRTCCTL_VSYNCENABLE
    '  00000800 - CRTCCTL_HSYNCENABLE
    '  00001000 - CRTCCTL_VIDEOENABLE
    '  00002000 - CRTCCTL_STEREOSCOPIC
    '  00004000 - CRTCCTL_FRAMEDISPLAYED
    '  00008000 - CRTCCTL_FRAMEBUFFERBGR
    '  00010000 - CRTCCTL_EVENFRAME
    '  00020000 - CRTCCTL_LINEDOUBLE
    '  00040000 - CRTCCTL_FRAMESWITCHED
    '
    ' Color formats:
    '  4  - 16 bpp (R5-G6-B5)
    '  5  - 16 bpp (R4-G4-B4-X4)
    '  12 - 32 bpp (R8-G8-B8-X8)
    VERITE_CRTCCTL = &H84

    ' Constant: VERITE_CRTCH
    ' Horizontal timing register
    VERITE_CRTCH = &H88

    ' Constant: VERITE_CRTCV
    ' Vertical timing register
    VERITE_CRTCV = &H8C
End Enum


Public Sub ModMain Cdecl Alias "main" ()
    Dim vram As Byte Ptr
    Dim cram As Byte Ptr
    Dim bios As Byte Ptr
    Dim vaddress as Byte Ptr

    vaddress = CPtr(Byte Ptr, &HA0000)
    vram = CPtr(Byte Ptr, &HA0000000)
    cram = CPtr(Byte Ptr, &HA0018000)
    bios = CPtr(Byte Ptr, &HA0020000)

    ' claim video memory
    memmap (64 * 1024 * 3) / 4096, vram, vaddress
    ' claim KBC
    portalloc &H60, 2
    ' claim VGA
    portalloc &H3C0, 32

    Dim lfb as Long
    Dim portbase as Long
    Dim chiptype as Long
    Dim chipsubtype as Long
    Dim bioshash as Long
    Dim lp as Long
    Dim devname As String

    ClearCon cram
    PrintString "BASIC For teh win (>O.O)>", cram, 25 * 80 - 27

    PrintString "Verite test program", cram, 0
    PrintString "Searching PCI device list for a Verite board...", cram, 80

    chiptype = Verite_PCIProbe(lfb, portbase)
    PrintString "Hashing BIOS...", cram, 160
    bioshash = 0
    for lp = 0 to 32767
        bioshash = bioshash + ((16381 * lp + 1) * CLng(bios[lp]))
    next lp
    PrintString "Probe result: " + hex$(chiptype), cram, 240
    If chiptype = 1 Or chiptype = 2 Then

        Select case bioshash
            Case &H4696828B
                devname = "Stealth II S220"
                chipsubtype = 1
            Case Else
                devname = "Unknown (0x" + hex$(bioshash) + ")"
                chipsubtype = 0
        End Select
        PrintString "Model: " + devname, cram, 400

        If chiptype = 1 Then
            devname = "Verite V1000"
        Else
            If chipsubtype = 2 Then
                devname = "Verite V2200"
            ElseIf chipsubtype = 1 Then
                devname = "Verite V2100"
            Else
                devname = "Verite V2x00"
            End If
        End If
        PrintString "GPU: " + devname, cram, 320


        PrintString "LFB Address: 0x" + hex$(lfb), cram, 480
        PrintString "Port Base: 0x" + hex$(portbase), cram, 560

        PrintString "Press Y to start the test at your hardware's peril", cram, 720
        PrintString "Press N to skip this test", cram, 800

        If GetYesNo() = 1 Then
            TestVerite(chiptype, chipsubtype, lfb, portbase)
        End If

        ClearCon cram
        PrintString "Test Ended. Please hit the power or the reset button", cram, 0

    Else
        PrintString "No Verite board found", cram, 320
        PrintString "Test Ended. Please hit the power or the reset button", cram, 400
    End if
    While 1 = 1

    Wend

End Sub

Public Sub PrintString (s As String, vram As Byte Ptr, offset As Integer)
    Dim lp As Long
    Dim ch As Byte

    For lp = 1 To len(s)
        ch = asc(mid$(s,lp,1))
        vram[lp * 2 - 2 + 2 * offset] = ch
        vram[lp * 2 - 1 + 2 * offset] = 7
    Next lp

End Sub

Public Function GetYesNo () As Integer
    ' Y = &H15
    ' N = &H31

    Dim status As Integer = -1
    Dim key As Integer
    While status = -1
        key = ReadKB
        if key = &H15 then status = 1
        if key = &H31 then status = 0
    Wend
    GetYesNo = status
End Function

Public Function ReadKB () as Integer
    Dim temp as Unsigned Byte
    Dim key as Unsigned Byte
    Dim breakexit as Unsigned Byte = 0
    Dim done as Unsigned Byte = 0
    Dim isescaped as Unsigned Byte = 0
    While done = 0
        key = inportb(&H60)
        temp = inportb(&H61)
        outportb(&H61,temp Or  &H80)    'disable
        outportb(&H61,temp And &H7F)    'and reenable
        If key = &HE0 Then
            isescaped = &H80
        ElseIf key > &H80 Then
            If key = breakexit Then
                done = 1
            Else
                breakexit = 0
                isescaped = 0
            End If
        Else
            breakexit = key or &H80
            ReadKB = key or isescaped
        End If
    Wend
End Function

Public Sub ClearCon (vram As Byte Ptr)
    Dim lp As Long
    For lp = 1 to 80 * 25 * 2
        vram[2 * lp - 2] = 0
        vram[2 * lp - 1] = 7
    Next lp
End Sub

Public Function Verite_PCIProbe(ByRef base_lfb As Long, ByRef base_pio As Integer) As Long

    ' load returns
    base_lfb = 0
    base_pio = 0

    ' find the pci manager
    Dim gate As Long
    gate = RouteFind(PORTNAME("PCI0"))
    If gate = 0 then
        Mach64_PCIProbe = &HE2
        Exit Function
    End If

    ' get the entry point
    Dim callback As Sub()
    Dim aspace As Long
    aspace = gatelookup(gate, @callback)

    ' find an card
    Dim arg1 As long
    Dim arg2 As long
    Dim arg3 As Long
    Dim pciaddr As Long
    Dim lp As Long
    arg2 = -1
    Verite_PCIProbe = &HE1
    Do
        ' iterate devices
        arg1 = MANGLE(PCI_ENUMERATEDEV, gate)
        Call Localipccall (callback, arg1, arg2, arg3, 0)
        If arg1 <> -1 then
            If arg1 = &H00011163 or arg1 = &H20001163 Then
                ' we found an Vxxxx board, get BARs and deduce framebuffer and port range
                If arg1 = &H00011163 then
                    Verite_PCIProbe = 1
                Else
                    Verite_PCIProbe = 2
                End if

                pciaddr = arg2
                For lp = 0 to 5
                    arg1 = MANGLE(PCI_GETBARTYPE, gate)
                    arg2 = pciaddr
                    arg3 = lp
                    Call Localipccall (callback, arg1, arg2, arg3, 0)
                    If arg2 = 0 Then
                        ' unused BAR
                    ElseIf (arg2 and &H1) = 1 Then
                        ' port range
                        arg1 = MANGLE(PCI_GETBARBASE, gate)
                        arg2 = pciaddr
                        arg3 = lp
                        Call Localipccall (callback, arg1, arg2, arg3, 0)
                        base_pio = arg2 And &HFFFE
                        
                        If base_pio > &H3000 Then
                            arg1 = MANGLE(PCI_SETBARBASE, gate)
                            arg2 = pciaddr
                            arg3 = lp
                            Call localipccall (callback, arg1, arg2, arg3, &H3000)
                            base_pio = &H3000
                        End if

                    ElseIf (0 - arg2 > (1024& * 1024&)) Then
                        ' this seems to be the lfb
                        arg1 = MANGLE(PCI_GETBARBASE, gate)
                        arg2 = pciaddr
                        arg3 = lp
                        Call Localipccall (callback, arg1, arg2, arg3, 0)
                        base_lfb = arg2 and &HFFFFFFF0
                    Else
                        ' some memory range with unknown purpose
                    End If

                Next lp

                ' break from the loop
                arg1 = -1
            End if
        End If

    Loop While arg1 <> -1

End Function

Sub PutPixel(vram As Byte Ptr, x As Long, y As Long, col As Long, bpp As Byte, pitch As Long)
    Dim offset As Long
    Select case bpp
        Case 4
            Dim temp as Byte
            offset = x \ 2 + y * pitch
            If (x And &H1) = 1 Then
                temp = vram[offset] and &HF
                vram[offset] = temp + 16 * (col and &HF)
            Else
                temp = vram[offset] and &HF0
                vram[offset] = temp + (col and &HF)
            End If

        Case 8
            offset = x + y * pitch
            vram[offset] = CByte(col)

        Case 16
            offset = 2 * x + pitch
            Dim mempos as Short Ptr
            mempos = CPtr(Short Ptr, @vram[offset])
            mempos[0] = CShort(col)

        Case 32
            offset = 4 * x + pitch
            Dim mempos as Long Ptr
            mempos = CPtr(Long Ptr, @vram[offset])
            mempos[0] = col

    End Select
End Sub

Sub WriteLfbString(lfb As Byte Ptr, s as String, offset as Long)
    Dim lp As Long
    Dim ch As Byte

    For lp = 1 To len(s)
        ch = asc(mid$(s,lp,1))
        lfb[lp * 8 - 8 + 8 * offset] = ch
        lfb[lp * 8 - 7 + 8 * offset] = &H1F
    Next lp
End Sub

Sub WriteConioString(vram As Byte Ptr, s as String, font() As Byte, row As Long, column As Long, col As Long, bpp as Byte, vw As Long)
    WriteGfxString vram, s, @font(0), column * 8, row * 16, col, bpp, vw
End Sub

Sub WriteGfxString(vram as Byte Ptr, s as String, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
    Dim lp as long
    For lp = 1 to len(s)
        WriteGfxChar vram, asc(mid$(s,lp,1)), font, x + 8 * (lp - 1), y, col, bpp, vw
    next lp
End Sub
Sub WriteGfxChar(vram as Byte Ptr, ch as Integer, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
    dim lp as long
    Dim fontbyte as byte
    For lp = 0 to 15
        fontbyte = font[32 * ch + lp]
        if (fontbyte and &H01) = &H01 then PutPixel vram, x + 7, y + lp, col, bpp, vw
        if (fontbyte and &H02) = &H02 then PutPixel vram, x + 6, y + lp, col, bpp, vw
        if (fontbyte and &H04) = &H04 then PutPixel vram, x + 5, y + lp, col, bpp, vw
        if (fontbyte and &H08) = &H08 then PutPixel vram, x + 4, y + lp, col, bpp, vw
        if (fontbyte and &H10) = &H10 then PutPixel vram, x + 3, y + lp, col, bpp, vw
        if (fontbyte and &H20) = &H20 then PutPixel vram, x + 2, y + lp, col, bpp, vw
        if (fontbyte and &H40) = &H40 then PutPixel vram, x + 1, y + lp, col, bpp, vw
        if (fontbyte and &H80) = &H80 then PutPixel vram, x + 0, y + lp, col, bpp, vw
    Next lp
End Sub

Sub RenderTest(vram as Byte Ptr, w as long, h as long, bpp as byte, vw as long)
    Dim lpx as long, lpy as long
    For lpx = 1 To w - 1 Step 2
        PutPixel vram, lpx, 0, 9, bpp, vw
        PutPixel vram, lpx, 1, 9, bpp, vw
        If lpx mod 10 = 9 then
            PutPixel vram, lpx, 2, 9, bpp, vw
        End If
        If lpx mod 40 = 39 then
            PutPixel vram, lpx, 3, 9, bpp, vw
        End If
        PutPixel vram, lpx, h-1, 9, bpp, vw
        PutPixel vram, lpx, h-2, 9, bpp, vw
    Next lpx
    For lpy = 1 To h - 1 Step 2
        PutPixel vram, 0, lpy, 9, bpp, vw
        PutPixel vram, 1, lpy, 9, bpp, vw

        PutPixel vram, w-1, lpy, 9, bpp, vw
        PutPixel vram, w-2, lpy, 9, bpp, vw

        If lpy mod 10 = 9 then
            PutPixel vram, w-3, lpy, 9, bpp, vw
        End If
        If lpy mod 40 = 39 then
            PutPixel vram, w-4, lpy, 9, bpp, vw
        End If
    Next lpy
End Sub

' Function: TestVerite
' runs the test suite for the Verite graphics card
'
' Parameters:
'     board_model   - the board type. 1 for a V1k, 2 for a V2k
'     board_version - the board rating.
'                     1 for a V2100, 2 for a V2200, 0 if unknown
'     lfb           - address of the framebuffer in memory
'     portbase      - the base I/O port for the Verite registers
'
Sub TestVerite(board_model As Long, board_version As Long, lfb As Long, portbase As Long)
    Dim vaddress as Byte Ptr
    Dim vram as Byte Ptr

    Dim lp as long
    Dim ws as String, ws2 as String

    ' claim board-specific addresses
    vaddress = CPtr(Byte Ptr, lfb)
    vram = CPtr(Byte Ptr, &HB0000000)
    memmap (16 * 1024 * 1024) / 4096, vram, vaddress
    portalloc portbase, &H100

    outportb(portbase + VREGS.VERITE_MEMENDIAN, 1)

    For lp = 0 to 80 * 25 * 8 step 8
        vram[lp+0] = asc("B")
    Next lp

    outportb(portbase + VREGS.VERITE_MEMENDIAN, 0)

    For lp = 0 to 80 * 25 * 8 step 8
        vram[lp+0] = &H20
        vram[lp+1] = &H1F
    Next lp

    WriteLfbString(vram, "Set System Byte ordering", 0)

    lp = inportb(portbase + VREGS.VERITE_GPUCTL)
    ws = ""
    if (lp and &H1) = &H1 then ws2 = ws + " Reset": ws = ws2
    if (lp and &H2) = &H2 then ws2 = ws + " Hold" : ws = ws2
    if (lp and &H4) = &H4 then ws2 = ws + " Debug": ws = ws2
    ws2 = "GPU Control: [" + ws + " ]"
    WriteLfbString vram, ws2, 80

    ReadKB

    lp = inportd(portbase + VREGS.VERITE_CRTCCTL)
    WriteLfbString vram, "CRTC Control: 0x" + hex$(lp), 160

    ReadKB

    lp = inportd(portbase + VREGS.VERITE_CRTCH)
    WriteLfbString vram, "CRTC Horizontal: 0x" + hex$(lp), 240

    ReadKB

    lp = inportd(portbase + VREGS.VERITE_CRTCV)
    WriteLfbString vram, "CRTC Vertical: 0x" + hex$(lp), 320

    ReadKB

    ' H sync: (from vga 640x480)
    ' Active = 80 character cells -> 0x4F
    ' Left = 3 character cells -> 0x2
    ' Right = 5 character cells -> 0x4
    ' Sync = 12 character cells ->  0xA
    outportd portbase + VREGS.VERITE_CRTCH, &H004A084F

    ' V sync:
    ' Active = 480 -> 0x1DF
    ' Top = 25 -> 0x18
    ' Bottom = 3 -> 0x2
    ' Sync = 2 -> 0x1
    outportd portbase + VREGS.VERITE_CRTCV, &H018211DF

    ' native mode
    outportb portbase + VREGS.VERITE_MODE, 0

    ' 32 bpp, enable output
    outportd portbase + VREGS.VERITE_CRTCCTL, &H1C0B

    ReadKB

    ' VGA mode
    outportb portbase + VREGS.VERITE_MODE, 2
End Sub